import React from 'react'
import ReactDOM from 'react-dom/client'
import studio from '@theatre/studio'
import Extension from '@/components/Extension'
import PublishExtension from '@/components/PublishExtension'
import { ExtensionScriptManager, ScriptLoadingType } from '@/ExtensionScriptManager'
import { extensionEntry, createExtensionSDK } from '@/scripts'
import { PlayCanvasTheatreEditor } from '@/theatre/PlayCanvasTheatreEditor'
import { insertAfter } from '@/utils/common'
import '@/global.css'
import '@/init'
import { pcEditor } from '@/global'
import SceneSettingsButton from '@/components/SceneSettingsButton'
import LoginButton from '@/components/LoginButton'
import useMainStore, { MainConfig } from '@/hooks/useMainStore'
import useQuestStore from '@/hooks/useQuestStore'
import { GLTFExporter2 } from './gltf-exporter'
import {
  seatHint,
  seatHintHover,
  seatHintFarAway,
  minimapArrow,
  minimapCone,
  minimapFrame,
  minimapMask,
} from '@/assets/preview-image'
import { cloneDeep } from 'lodash'

// initialize the theatre studio at the first time is needed, but we choose to hide it by default
studio.initialize()
studio.ui.hide()

let currentUrl = window.location.href
let catchCopySource: any[] = []

const IMAGE_ASSET_CONFIGS = [
  { atlasName: 'seat-hint-atlas.png', spriteName: 'seat-hint-sprite', url: seatHint },
  {
    atlasName: 'seat-hover-hint-atlas.png',
    spriteName: 'seat-hover-hint-sprite',
    url: seatHintHover,
  },
  {
    atlasName: 'seat-hint-far-away-atlas.png',
    spriteName: 'seat-hint-far-away-sprite',
    url: seatHintFarAway,
  },
]

const IMAGE_ASSET_TEXTURE_CONFIGS = [
  { imgName: 'minimap-arrow.png', domId: 'minimap-arrow', url: minimapArrow },
  { imgName: 'minimap-cone.png', domId: 'minimap-cone', url: minimapCone },
  { imgName: 'minimap-frame.png', domId: 'minimap-frame', url: minimapFrame },
  { imgName: 'minimap-mask.png', domId: 'minimap-mask', url: minimapMask },
]

function resetExtensionEntity() {
  setTimeout(async () => {
    if (currentUrl !== window.location.href) {
      console.log('[Reset extension entity]:', { currentUrl, newUrl: window.location.href })
      currentUrl = window.location.href

      const { reloadExtensionConfigs } = useMainStore.getState()
      const { reloadQuestConfigs } = useQuestStore.getState()

      await handleExtensionEntity()

      reloadExtensionConfigs()
      reloadQuestConfigs()
    }
  }, 1000)
}

async function handleExtensionManager() {
  const extensionScriptManager = new ExtensionScriptManager()
  await extensionScriptManager.initial()

  await extensionScriptManager.uploadScript('extension-entry@v2', extensionEntry)

  await extensionScriptManager.uploadCreateSDKScript({
    filename: `create-extensions-sdk.mjs`, // ESM module
    text: createExtensionSDK,
    preload: false,
    data: {
      scripts: {},
      loading: false,
      loadingType: ScriptLoadingType.BeforeEngine,
    },
  })

  const createAtlasAndSpriteAsset = async (image: {
    atlasName: string
    spriteName: string
    url: string
  }): Promise<void> => {
    return extensionScriptManager.createTextureAtlas(image.atlasName, image.url).then((atlas) => {
      if (atlas) {
        extensionScriptManager.createSpriteAsset(image.spriteName, atlas)
      }
    })
  }

  const createTextureAsset = async (image: { imgName: string; url: string }): Promise<void> => {
    return extensionScriptManager.uploadImage(image.imgName, image.url).then((asset) => {
      if (asset) {
        console.log('[Successfully uploaded image asset]', { asset })
      }
    })
  }

  await Promise.all(IMAGE_ASSET_CONFIGS.map((image) => createAtlasAndSpriteAsset(image)))
  await Promise.all(IMAGE_ASSET_TEXTURE_CONFIGS.map((image) => createTextureAsset(image)))
}

async function handleExtensionEntity() {
  const existedExtensionEntity = pcEditor.entities.root.findByName('ExtensionEntity')
  const extensionScriptLoaded = pcEditor.call('assets:scripts:list').includes('extensionEntry')

  if (!existedExtensionEntity && extensionScriptLoaded) {
    window.extensionEntity = pcEditor.entities.create({
      parent: pcEditor.entities.root,
      name: 'ExtensionEntity',
    })

    try {
      await window.extensionEntity.addScript('extensionEntry')
    } catch (error) {
      // this is caused by delete scene of current scene
      // so we need to reload the page to prevent the error
      console.log('[Failed to add script to extension entity]', {
        error,
      })
      location.reload()
    }
  } else if (existedExtensionEntity && window.extensionEntity !== existedExtensionEntity) {
    window.extensionEntity = existedExtensionEntity
  } else {
    console.error('[Failed to create extension entity]')
  }
}

function renderLoginButtonUI() {
  const app = document.createElement('div')
  app.id = 'login-button-root'

  const layoutAttributes = document.querySelector('#layout-toolbar')
  layoutAttributes?.append(app)

  console.log('[LOGIN BUTTON INJECTION]', { layoutAttributes })

  ReactDOM.createRoot(app).render(
    <React.StrictMode>
      <LoginButton />
    </React.StrictMode>,
  )
}

function renderSceneSettingButtonUI() {
  const app = document.createElement('div')
  app.id = 'scene-settings-button-root'

  const layoutAttributes = document.querySelector('#layout-toolbar')
  layoutAttributes?.append(app)

  console.log('[SCENE SETTING BUTTON INJECTION]', { layoutAttributes })

  ReactDOM.createRoot(app).render(
    <React.StrictMode>
      <SceneSettingsButton />
    </React.StrictMode>,
  )
}

function renderExtensionUI() {
  const app = document.createElement('div')
  app.id = 'extension-app-root'

  const layoutAttributes = document.querySelector('#layout-attributes')
  layoutAttributes?.append(app)

  console.log('[CUSTOM UI INJECTION]', { layoutAttributes })

  ReactDOM.createRoot(app).render(
    <React.StrictMode>
      <Extension />
    </React.StrictMode>,
  )
}

function renderPublishUI() {
  const publishEntryRef = document.querySelector(
    '.picker-builds-publish .publish-buttons-container',
  )
  const publishList = publishEntryRef?.parentElement?.querySelector('.builds-list-heading')

  console.log('[PUBLISH UI INJECTION]', { publishEntryRef, publishList })

  if (publishEntryRef) {
    if (publishList) {
      publishList.innerHTML = 'PlayCanvas Existing Builds'
    }

    const customPublishRoot = document.createElement('div')
    customPublishRoot.id = 'custom-publish-root'
    insertAfter(customPublishRoot, publishEntryRef)

    ReactDOM.createRoot(customPublishRoot).render(
      <React.StrictMode>
        <PublishExtension />
      </React.StrictMode>,
    )
  }
}

function overrideEditor() {
  // Override the source code of PlayCanvas Editor to add event.
  return new Promise<void>((resolve, reject) => {
    try {
      pcEditor.methodRemove('entities:duplicate')
      const projectUserSettings = pcEditor.call('settings:projectUser')
      /**
       * Duplicates the specified entities and adds them to the scene.
       *
       * @param {Observer[]} entities - The entities to duplicate
       */
      pcEditor.method('entities:duplicate', function (entities: any[]) {
        pcEditor.entities.duplicate(
          entities.map((entity) => entity.apiEntity),
          {
            select: true,
            history: true,
            rename: projectUserSettings.get('editor.renameDuplicatedEntities'),
          },
        )
        pcEditor.emit('entities:duplicate', entities)
      })

      pcEditor.methodRemove('entities:copy')
      /**
       * Copies the specified entities into localStorage
       *
       * @param {Observer[]} entities - The entities to copy
       */
      pcEditor.method('entities:copy', function (entities: any[]) {
        pcEditor.entities.copyToClipboard(entities.map((e) => e.apiEntity))
        pcEditor.emit('entities:copy', entities)
      })

      pcEditor.methodRemove('entities:paste')
      /**
       * Pastes entities in localStore under the specified parent
       *
       * @param {Observer} parent - The parent entity
       */
      pcEditor.method('entities:paste', function (parent: any) {
        pcEditor.entities.pasteFromClipboard(parent ? parent.apiEntity : null)
        pcEditor.emit('entities:paste', catchCopySource)
      })
      resolve()
    } catch (error) {
      reject(error)
    }
  })
}

function combineMaps(
  combinedMap: Map<string, string[]>,
  map1: Map<string, string>,
  map2: Map<string, string>,
) {
  // const combinedMap = new Map()

  // Helper function to add values to the map
  const addToMap = (map: Map<string, any[]>, key: string, value: any) => {
    if (map.has(key)) {
      map.get(key)?.push(value)
    } else {
      map.set(key, [value])
    }
  }

  // Add all entries from map1
  for (const [key, value] of map1) {
    addToMap(combinedMap, key, value)
  }

  // Add all entries from map2
  for (const [key, value] of map2) {
    addToMap(combinedMap, key, value)
  }

  return combinedMap
}

function copyEntityStructure(source: any) {
  const result = new Map<string, string>()
  result.set(source.get('name'), source.get('resource_id'))
  function recurse(currentObj: pc.Entity) {
    if (currentObj.children && currentObj.children.length > 0) {
      currentObj.children.forEach((child: any) => {
        if (child._guid) {
          result.set(`${currentObj.name}_${child.name}`, child._guid)
          recurse(child)
        }
      })
    }
  }

  recurse(source.entity || source.viewportEntity)
  return result
}

function hasEntityPicker(config: MainConfig) {
  const string = JSON.stringify(config)
  return string.includes('entityId')
}

function updateEntityId(combinedMap: Map<string, any[]>, index: number, copy: any) {
  // Recursive function to traverse and update entityId
  function traverseAndUpdate(obj: any) {
    for (const key in obj) {
      if (typeof obj[key] === 'object' && obj[key] !== null) {
        traverseAndUpdate(obj[key])
      } else if (key === 'entityId') {
        const searchValue = obj[key]
        for (const [_, value] of combinedMap) {
          // Check if the current value array includes the search value
          if (value.includes(searchValue)) {
            obj[key] = value[index]
          }
        }
      }
    }
  }

  // Start the traversal
  traverseAndUpdate(copy)
}

function copyExtensionConfigToNewEntity(sourceEntities: any[]) {
  catchCopySource = sourceEntities
  const config = JSON.parse(
    window.extensionEntity.get(
      'components.script.scripts.extensionEntry.attributes.configs.main',
    ) || '{}',
  )

  const copyEntities = pcEditor.selection.items // A list of duplicated entities.
  sourceEntities.forEach((source: any) => {
    let combinedMap = new Map()
    const originMap = copyEntityStructure(source)

    const copyList = copyEntities.filter(
      (entity: any) => entity.viewportEntity.name === source.get('name'),
    )

    copyList.forEach((copy: any) => {
      const copyMap = copyEntityStructure(copy)
      combinedMap = combineMaps(combinedMap, originMap, copyMap)
    })

    for (const [_, value] of combinedMap.entries()) {
      const [originEntityId, ...duplicatedId] = value
      if (config[originEntityId]) {
        duplicatedId.forEach((guid: string, index: number) => {
          const deepCopy = cloneDeep(config[originEntityId])
          if (hasEntityPicker(deepCopy)) {
            updateEntityId(combinedMap, index + 1, deepCopy)
          }
          config[guid] = deepCopy
        })
      }
    }
  })

  window.extensionEntity.set(
    'components.script.scripts.extensionEntry.attributes.configs.main',
    JSON.stringify(config),
  )
}

function onReady() {
  if (!pcEditor) return

  console.log('[Successfully got window.editor]', { editor: pcEditor })

  pcEditor.once('assets:load', async () => {
    bindGLTFExportHotKey()

    // Add the extension scripts, and smartly handle the version with the file naming
    await handleExtensionManager()
    // Create the extension entity
    await handleExtensionEntity()

    // Initial Theatre editor
    const playCanvasTheatreEditor = new PlayCanvasTheatreEditor()
    console.log({ playCanvasTheatreEditor })

    renderSceneSettingButtonUI()
    renderLoginButtonUI()
    renderExtensionUI()
    renderPublishUI()
  })

  pcEditor.on('entities:load', handleExtensionEntity)
  pcEditor.on('scene:unload', resetExtensionEntity)

  overrideEditor()
    .then(() => {
      pcEditor.on('entities:duplicate', copyExtensionConfigToNewEntity)
      pcEditor.on('entities:copy', copyExtensionConfigToNewEntity)
      pcEditor.on('entities:paste', copyExtensionConfigToNewEntity)
    })
    .catch((error) => {
      console.error(error)
      console.warn(
        "The Creator tools's copy/paste and duplicate functions currently do not work with the Extension config.",
      )
    })
}

if (document.readyState !== 'loading') {
  onReady() // Or setTimeout(onReady, 0); if you want it consistently async
} else {
  document.addEventListener('DOMContentLoaded', onReady)
}

const bindGLTFExportHotKey = () => {
  document.addEventListener('keypress', async (e) => {
    if (e.altKey && e.code === 'KeyD') {
      new GLTFExporter2().export()
    }
  })
}
